/*
 * HomologHMM 1.04
 * (c) Lukas Kll
 * Distributable under GPL license.
 * See terms of license at gnu.org.
 */
package se.ki.cgb.anhmmfile;

import java.io.*;
import java.util.*;
import org.biojava.bio.*;
import org.biojava.bio.symbol.*;
import org.biojava.bio.seq.*;
import org.biojava.bio.seq.io.*;

/**
 * Desribes the Labeled fasta format
 * @author Lukas.Kall@cgb.ki.se
 * @version $Revision: 1.4 $
 */
public class LabeledFastaFormat extends FastaFormat {

	public static final String DEFAULT = "LABELEDFASTA";
	/**
	 * Constant string which is the property key used to notify
	 * listeners of the label lines of FASTA sequences.
	 */
	public final static String PROPERTY_LABELS = "labels";

	public static final long serialVersionUID = 234234;
	
	/**
	 * 
	 */
	public LabeledFastaFormat() {
		super();
	}

	public boolean readSequence(
	  BufferedReader reader,
		  SymbolTokenization symParser,
		  SeqIOListener siol
	)	throws
	  IllegalSymbolException,
	  IOException,
	  ParseException
	{
	  String line = reader.readLine();
	  if (line == null) {
		throw new IOException("Premature stream end");
	  }
	  while(line.length() == 0) {
		line = reader.readLine();
		if (line == null) {
		  throw new IOException("Premature stream end");
		}
	  }
	  if (!line.startsWith(">")) {
		throw new IOException("Stream does not appear to contain FASTA formatted data: " + line);
	  }
    
	  siol.startSequence();
    
	  String description = line.substring(1).trim();
	  siol.addSequenceProperty(PROPERTY_DESCRIPTIONLINE, description);
    
	  String name = new java.util.StringTokenizer(description).nextToken();
	  siol.setName(name);
    
	  boolean seenEOF = readSequenceData(reader, symParser, siol);
	  siol.endSequence();
    
	  return !seenEOF;
	}

 	private boolean readSequenceData(
	  BufferedReader r,
	  SymbolTokenization parser,
	  SeqIOListener listener
	) throws
	  IOException,
	  IllegalSymbolException
//	  ,ParseException
	{
	  char[] cache = new char[512];
	  boolean reachedEnd = false, seenEOF = false;
	  String parsingLabel="";
	  StreamParser sparser = parser.parseStream(listener);
	  Hashtable labels = new Hashtable();
	  while (!reachedEnd) {
		r.mark(cache.length + 1);
		int bytesRead = r.read(cache, 0, cache.length);
		if (bytesRead < 0) {
		  reachedEnd = seenEOF = true;
		} else {
		  int parseStart = 0;
		  int parseEnd = 0;
		  while (!reachedEnd && parseStart < bytesRead && cache[parseStart] != '>') {
			parseEnd = parseStart;
          
			while (parseEnd < bytesRead &&
			  cache[parseEnd] != '\n' &&
			  cache[parseEnd] != '\r'
			) {
			  ++parseEnd;
			}
            if (parsingLabel !=""|| cache[parseStart]=='#' || cache[parseStart]=='?') {
            	parsingLabel = parseLabel(parsingLabel,labels,cache, parseStart, parseEnd);
            } else {
//				sparser.characters(cache, parseStart, parseEnd - parseStart);
				int pos=parseStart;

				while(pos<parseEnd) {
					if (cache[pos]!=' ')
						sparser.characters(cache, pos, 1);
					pos++;
				}
            }
          
			parseStart = parseEnd + 1;
			while (parseStart < bytesRead &&
			  (cache[parseStart] == '\n' ||
			   cache[parseStart] == '\r') )
			  {
				++parseStart;
			  }
		  }
		  if (parseStart < bytesRead && cache[parseStart] == '>') {
			try {
			  r.reset();
			} catch (IOException ioe) {
			  throw new IOException(
				"Can't reset: " +
				ioe.getMessage() +
				" parseStart=" + parseStart +
				" bytesRead=" + bytesRead
			  );
			}
			if (r.skip(parseStart) != parseStart) {
			  throw new IOException("Couldn't reset to start of next sequence");
			}
			reachedEnd = true;
			parsingLabel ="";
		  }
		}
	  }
    
	  try {
		listener.addSequenceProperty(PROPERTY_LABELS, labels);
	  } catch (ParseException e) {
		System.err.println("Could not set sequence property");
		e.printStackTrace();
	  }
	  sparser.close();
	  return seenEOF;
	}

	protected String parseLabel(String key,Hashtable labels, char[] cache, int parseStart, int parseEnd) {
        int pos=parseStart;
        String label="";
        if (key=="" || !labels.containsKey(key)) {
			while(cache[pos]!=' '){
				key += cache[pos];
				pos++;
				if (pos>=parseEnd)
				    return key;			
			}
        }
		while(pos<parseEnd) {
			if (cache[pos]!=' ')
				label += cache[pos];
				pos++;
		}
		if (labels.containsKey(key)) {
			String newLabel = (String) labels.get(key) + label;
			labels.put(key,newLabel);
		} else {
			labels.put(key,label);
		}
		return key;
	}

	public void writeSequence(Sequence seq, PrintStream os)
	throws IOException {
		int len = seq.length();
		SymbolTokenization tokenization = null;
		try {
			tokenization = ProteinTools.getAlphabet().getTokenization("token");
		}
        catch (BioException e) {
        }
		Hashtable labels = getLabels(seq);

		os.print(">");
		os.println(describeSequence(seq));
		for(int i = 0; i <= len/lineWidth; i++) {
			os.print("  ");
			try {
				for(int j=i*lineWidth; j<Math.min((i+1)*lineWidth, len); j++)  {
					if ( j%10 == 0) os.print(" ");
					os.print(tokenization.tokenizeSymbol(seq.symbolAt(j+1)));
				}
				os.println();
			} catch (IllegalSymbolException e) {
				os.println();
			}
			if (labels != null && labels.size()>0) {
				for (Enumeration keys = labels.keys();keys.hasMoreElements();) {
					String key = (String) keys.nextElement();
					String lab = (String) labels.get(key);
					while (key.length()<2) key += ' ';
					os.print(key);
					for(int j=i*lineWidth; j<Math.min((i+1)*lineWidth, len); j++)  {
						if ( j%10 == 0) os.print(" ");
						os.print(lab.charAt(j));
					}
					os.println();
				}
			}
			os.println();
		}    	
 	}

	public static Hashtable getLabels(Sequence seq) {
	    Hashtable labels = null;
		Annotation seqAnn = seq.getAnnotation();

		if(seqAnn.containsProperty(PROPERTY_LABELS)) {
		  labels = (Hashtable) seqAnn.getProperty(PROPERTY_LABELS);
		} 
	    return labels;
	}

	public static void setLabels(Sequence seq,Hashtable labels) throws Exception {
		Annotation seqAnn = seq.getAnnotation();

		if(seqAnn.containsProperty(PROPERTY_LABELS)) {
		  seqAnn.setProperty(PROPERTY_LABELS,labels);
		} 
	}
}
